/******************************************************************
 *FILE: atmel_mxt_ts.h
 *SW-COMPONENT: QNX atmel touch driver
 *DESCRIPTION: This is header file for the atmel core driver.
 *COPYRIGHT: © 2019 Robert Bosch GmbH
 *
 *This program is free software; you can redistribute  it and/or modify it
 *under  the terms of  the GNU General  Public License as published by the
 *Free Software Foundation;  either version 2 of the  License, or (at your
 *option) any later version.
 ******************************************************************/
#ifndef ATMEL_MXT_TS_H_
#define ATMEL_MXT_TS_H_

#ifdef  __cplusplus
extern "C" {
#endif
#include <sys/types.h>
#include <stdbool.h>

#include "atmel_mxt_i2c.h"
#include "fidm_touch.h"

#ifndef UNITTEST
#include <sys/neutrino.h>
#include <sys/dispatch.h>
#include <screen/screen.h>
#include "dcmd_pm.h"
#else
#include "screen_header.h"
#include "ipc_header.h"
#include <errno.h>
#endif

#define PM_DEV_NODE					"/dev/pm"

#define ATMEL_DEV_NAME				"atmel"
#define BIT(x)						( 1 << x )

#define swap(x,y) \
		do {\
		x = x ^ y;\
		y = x ^ y;\
		x = x ^ y;\
		} while(0)

#define RTD_REPORTID_COUNT                    5
#define T5_MSG_SIZE                           10
#define T5_LAST_MSG_SIZE                      5
#define MXT_PULSE_PRIORITY                    21
#define MXT_PULSE_CODE                        _PULSE_CODE_MINAVAIL
#define MXT_RESET_PULSE_CODE                  (MXT_PULSE_CODE+1)
#define MXT_POWERMAN_UPDATE_PULSE_PRIORITY    -1
#define MXT_POWERMAN_UPDATE_PULSE_CODE        (MXT_PULSE_CODE+2)
#define MXT_POWERMAN_CONNECT_PULSE_PRIORITY   -1
#define MXT_POWERMAN_CONNECT_PULSE_CODE       (MXT_PULSE_CODE+3)
#define MXT_WATCHDOG_PULSE_CODE               (MXT_PULSE_CODE+4)

/* Pulse codes for receiving pulses from power manager */
#define MXT_PM_PREPARE_PULSE 				  (_PULSE_CODE_MINAVAIL+5)
#define MXT_PM_SUSPEND_PULSE 				  (_PULSE_CODE_MINAVAIL+6)
#define MXT_PM_RESUME_PULSE  				  (_PULSE_CODE_MINAVAIL+7)

typedef enum {
	MXT_CTRL_MSG_TYPE_XY_DATA,
	MXT_CTRL_MSG_TYPE_LASTXY_DATA
} mxt_ctrl_msg_type;

typedef enum {
  FIRST_MSG_PKT = 0,
  SECOND_MSG_PKT,
  THIRD_MSG_PKT,
  FOURTH_MSG_PKT,
  LAST_MSG_PKT
} t33_msg_pkt;

/*
 * Get the last x/y coordinates that were sent to screen/application
 */
typedef struct {
	mxt_ctrl_msg_type type;
	void* buf;
	unsigned buflen;
} mxt_ctrl_msg_xy_data;

typedef union {
	struct _pulse pulse;
	mxt_ctrl_msg_type type;
	mxt_ctrl_msg_xy_data xy_data;
} mxt_ctrl_msg;

/* Registers */
#define MXT_OBJECT_START			0x07
#define MXT_OBJECT_SIZE				6
#define MXT_INFO_CHECKSUM_SIZE		3
#define MXT_MAX_BLOCK_WRITE			255
#define MXT_INFO_BASE_REG			0

/* Object types */
#define MXT_DEBUG_DIAGNOSTIC_T37	37
#define MXT_GEN_MESSAGE_T5			5
#define MXT_GEN_COMMAND_T6			6
#define MXT_GEN_POWER_T7			7
#define MXT_GEN_ACQUIRE_T8			8
#define MXT_GEN_DATASOURCE_T53		53
#define MXT_TOUCH_MULTI_T9			9
#define MXT_TOUCH_KEYARRAY_T15		15
#define MXT_TOUCH_PROXIMITY_T23		23
#define MXT_TOUCH_PROXKEY_T52		52
#define MXT_PROCI_GRIPFACE_T20		20
#define MXT_PROCG_NOISE_T22			22
#define MXT_PROCI_ONETOUCH_T24		24
#define MXT_PROCI_TWOTOUCH_T27		27
#define MXT_PROCI_GRIP_T40			40
#define MXT_PROCI_PALM_T41			41
#define MXT_PROCI_TOUCHSUPPRESSION_T42	42
#define MXT_PROCI_STYLUS_T47		47
#define MXT_PROCG_NOISESUPPRESSION_T48	48
#define MXT_SPT_COMMSCONFIG_T18		18
#define MXT_SPT_GPIOPWM_T19			19
#define MXT_SPT_SELFTEST_T25		25
#define MXT_SPT_CTECONFIG_T28		28
#define MXT_SPT_USERDATA_T38		38
#define MXT_SPT_DIGITIZER_T43		43
#define MXT_SPT_MESSAGECOUNT_T44	44
#define MXT_SPT_CTECONFIG_T46		46
#define MXT_SPT_DYNAMICCONFIGURATIONCONTAINER_T71 71
#define MXT_PROCI_SYMBOLGESTUREPROCESSOR	92
#define MXT_PROCI_TOUCHSEQUENCELOGGER	93
#define MXT_TOUCH_MULTITOUCHSCREEN_T100 100
#define MXT_PROCI_ACTIVESTYLUS_T107	107
#define MXT_PROCI_KNOBDATA_T152	    152
#define MXT_TOUCH_DIAGNOSTICS_T33	33
#define MXT_RPTID_RESERVED			0

/* MXT_GEN_MESSAGE_T5 object */
#define MXT_RPTID_NOMSG		0xff

/* MXT_GEN_COMMAND_T6 field */
#define MXT_COMMAND_RESET		0
#define MXT_COMMAND_BACKUPNV	1
#define MXT_COMMAND_CALIBRATE	2
#define MXT_COMMAND_REPORTALL	3
#define MXT_COMMAND_DIAGNOSTIC	5

/* Define for T6 status byte */
#define MXT_T6_STATUS_RESET		BIT(7)
#define MXT_T6_STATUS_OFL		BIT(6)
#define MXT_T6_STATUS_SIGERR	BIT(5)
#define MXT_T6_STATUS_CAL		BIT(4)
#define MXT_T6_STATUS_CFGERR	BIT(3)
#define MXT_T6_STATUS_COMSERR	BIT(2)

/* MXT_GEN_POWER_T7 field */
struct t7_config {
	uint8_t idle;
	uint8_t active;
} __attribute__ ((packed));

#define MXT_POWER_CFG_RUN		0
#define MXT_POWER_CFG_DEEPSLEEP		1

/* MXT_TOUCH_MULTI_T9 field */
#define MXT_T9_CTRL		0
#define MXT_T9_XSIZE		3
#define MXT_T9_YSIZE		4
#define MXT_T9_ORIENT		9
#define MXT_T9_RANGE		18

/* MXT_TOUCH_MULTI_T9 status */
#define MXT_T9_UNGRIP		BIT(0)
#define MXT_T9_SUPPRESS		BIT(1)
#define MXT_T9_AMP			BIT(2)
#define MXT_T9_VECTOR		BIT(3)
#define MXT_T9_MOVE			BIT(4)
#define MXT_T9_RELEASE		BIT(5)
#define MXT_T9_PRESS		BIT(6)
#define MXT_T9_DETECT		BIT(7)

struct t9_range {
	uint16_t x;
	uint16_t y;
} __attribute__ ((packed));

/* MXT_TOUCH_MULTI_T9 orient */
#define MXT_T9_ORIENT_SWITCH	BIT(0)
#define MXT_T9_ORIENT_INVERTX	BIT(1)
#define MXT_T9_ORIENT_INVERTY	BIT(2)

/* MXT_SPT_COMMSCONFIG_T18 */
#define MXT_COMMS_CTRL		0
#define MXT_COMMS_CMD		1
#define MXT_COMMS_RETRIGEN      BIT(6)

/* MXT_DEBUG_DIAGNOSTIC_T37 */
#define MXT_DIAGNOSTIC_PAGEUP	0x01
#define MXT_DIAGNOSTIC_DELTAS	0x10
#define MXT_DIAGNOSTIC_REFS	0x11
#define MXT_DIAGNOSTIC_SIZE	128

#define MXT_FAMILY_1386			160
#define MXT1386_COLUMNS			3
#define MXT1386_PAGES_PER_COLUMN	8

struct t37_debug {
#ifdef CONFIG_TOUCHSCREEN_ATMEL_MXT_T37
	uint8_t mode;
	uint8_t page;
	uint8_t data[MXT_DIAGNOSTIC_SIZE];
#endif
};

/* Define for MXT_GEN_COMMAND_T6 */
#define MXT_BOOT_VALUE		0xa5
#define MXT_RESET_VALUE		0x01
#define MXT_BACKUP_VALUE	0x55

/* Define for MXT_PROCI_TOUCHSUPPRESSION_T42 */
#define MXT_T42_MSG_TCHSUP	BIT(0)

/* T100 Multiple Touch Touchscreen */
#define MXT_T100_CTRL		0
#define MXT_T100_CFG1		1
#define MXT_T100_TCHAUX		3
#define MXT_T100_XSIZE		9
#define MXT_T100_XRANGE		13
#define MXT_T100_YSIZE		20
#define MXT_T100_YRANGE		24

#define MXT_T100_CFG_SWITCHXY	BIT(5)
#define MXT_T100_CFG_INVERTY	BIT(6)
#define MXT_T100_CFG_INVERTX	BIT(7)

#define MXT_T100_TCHAUX_VECT	BIT(0)
#define MXT_T100_TCHAUX_AMPL	BIT(1)
#define MXT_T100_TCHAUX_AREA	BIT(2)

#define MXT_T100_DETECT		BIT(7)
#define MXT_T100_TYPE_MASK	0x70

enum t100_type {
	MXT_T100_TYPE_FINGER		= 1,
	MXT_T100_TYPE_PASSIVE_STYLUS	= 2,
	MXT_T100_TYPE_ACTIVE_STYLUS	= 3,
	MXT_T100_TYPE_HOVERING_FINGER	= 4,
	MXT_T100_TYPE_GLOVE		= 5,
	MXT_T100_TYPE_LARGE_TOUCH	= 6,
};

#define MXT_DISTANCE_ACTIVE_TOUCH	0
#define MXT_DISTANCE_HOVERING		1

#define MXT_TOUCH_MAJOR_DEFAULT		1
#define MXT_PRESSURE_DEFAULT		1

/* Gen2 Active Stylus */
#define MXT_T107_STYLUS_STYAUX		42
#define MXT_T107_STYLUS_STYAUX_PRESSURE	BIT(0)
#define MXT_T107_STYLUS_STYAUX_PEAK	BIT(4)

#define MXT_T107_STYLUS_HOVER		BIT(0)
#define MXT_T107_STYLUS_TIPSWITCH	BIT(1)
#define MXT_T107_STYLUS_BUTTON0		BIT(2)
#define MXT_T107_STYLUS_BUTTON1		BIT(3)

/* Delay times */
#define MXT_BACKUP_TIME		50	/* msec */
#define MXT_RESET_GPIO_TIME	20	/* msec */
#define MXT_RESET_INVALID_CHG	100	/* msec */
#define MXT_RESET_TIME		200	/* msec */
#define MXT_RESET_TIMEOUT	3000	/* msec */
#define MXT_CRC_TIMEOUT		1000	/* msec */
#define MXT_FW_RESET_TIME	3000	/* msec */
#define MXT_FW_CHG_TIMEOUT	300	/* msec */
#define MXT_WAKEUP_TIME		25	/* msec */
#define MXT_REGULATOR_DELAY	150	/* msec */
#define MXT_CHG_DELAY	        100	/* msec */
#define MXT_POWERON_DELAY	150	/* msec */
#define MXT_BOOTLOADER_WAIT	36E5	/* 1 minute */
#define MXT_WATCHDOG_TIMEOUT	1000	/* msec */
#define MXT_CONFIG_TIMEOUT	1000	/* msec */

/* Command to unlock bootloader */
#define MXT_UNLOCK_CMD_MSB	0xaa
#define MXT_UNLOCK_CMD_LSB	0xdc

/* Bootloader mode status */
#define MXT_WAITING_BOOTLOAD_CMD	0xc0	/* valid 7 6 bit only */
#define MXT_WAITING_FRAME_DATA	0x80	/* valid 7 6 bit only */
#define MXT_FRAME_CRC_CHECK	0x02
#define MXT_FRAME_CRC_FAIL	0x03
#define MXT_FRAME_CRC_PASS	0x04
#define MXT_APP_CRC_FAIL	0x40	/* valid 7 8 bit only */
#define MXT_BOOT_STATUS_MASK	0x3f
#define MXT_BOOT_EXTENDED_ID	BIT(5)
#define MXT_BOOT_ID_MASK	0x1f

/* Touchscreen absolute values */
#define MXT_MAX_AREA		0xff

#define MXT_PIXELS_PER_MM	20

#define DEBUG_MSG_MAX		200
#define MXT_MAX_NUM_TOUCHPTS	10
#define MXT_MAX_RETRY_ATTEMPTS	2
#define MXT_MAX_RETRY_ATTEMPTS_ON_RESUME	10

enum device_state {
	MXT_STATE_READY,
	MXT_STATE_UPDATING_CONFIG,
	MXT_STATE_UPDATING_CONFIG_ASYNC,
	MXT_STATE_START,
	MXT_STATE_STOP,
	MXT_STATE_GOING_AWAY
};

enum mxt_cmd {
	UPDATE_CFG,
	UPDATE_FW
};

struct mxt_info {
	uint8_t family_id;
	uint8_t variant_id;
	uint8_t version;
	uint8_t build;
	uint8_t matrix_xsize;
	uint8_t matrix_ysize;
	uint8_t object_num;
};

struct mxt_object {
	uint8_t type;
	uint16_t start_address;
	uint8_t size_minus_one;
	uint8_t instances_minus_one;
	uint8_t num_report_ids;
} __attribute__ ((packed));

/* Config update context */
struct mxt_cfg {
	uint8_t *raw;
	size_t raw_size;
	off_t raw_pos;

	uint8_t *mem;
	size_t mem_size;
	int start_ofs;

	struct mxt_info info;
};

struct mxt_statusinfo {
	bool dev_status;
	bool intp_triggered;
	uint32_t error_count;
};

struct mxt_touch_state {
    uint8_t status;
    uint16_t x;
    uint16_t y;
    uint8_t pressure;
    uint8_t width;
    uint8_t height;
    uint8_t orientation;
};

struct mxt_msg_thread {
	pthread_t fidm_thread;
	name_attach_t *fidm_attach;
	int fidm_coid;
	char *fidm_attach_point;
};

/* Each client has this additional data */
struct mxt_data {
	/* Resource manager specific */
	pthread_t recv_thread;
	unsigned thread_priority;
	int chid;
	int coid;
	name_attach_t *attach;
	char *attach_point;

	struct mxt_msg_thread *msg_thread;

	/* mtouch framework */
	struct mtouch_device* inputevents_hdl;
	struct mxt_touch_state touch_state[MXT_MAX_NUM_TOUCHPTS];
	struct mxt_touch_state last_coords;
	uint32_t seq_id;
	unsigned width;
	unsigned height;
	unsigned invert_x;
	unsigned invert_y;
	unsigned swap_xy;

	/* Board Specific */
	unsigned reset_gpio_pin;
	unsigned tp_intr;
	uint32_t tp_intr_gpio;
	int tp_iid;
	struct sigevent tp_intrevent;

	/* I2C */
	int i2c_fd;
	char* i2c_devname;
	unsigned i2c_speed;
	unsigned i2c_slave_addr;
	unsigned i2c_reg_addr_len;
	atmel_libi2c_funcs_t i2c_funcs;

	/* pm */
	char* pm_dev_name;
	int mxt_pm_fd;
	enum pm_state mxt_pm_state;

	/* Info Block */
	struct mxt_object *object_table;
	struct mxt_info *info;

	/* Cached parameters from object table */
	uint16_t T5_address;
	uint8_t T5_msg_size;
	uint8_t T6_reportid;
	uint16_t T6_address;
	uint16_t T7_address;
	uint16_t T71_address;
	uint8_t T9_reportid_min;
	uint8_t T9_reportid_max;
	uint8_t T15_reportid_min;
	uint8_t T15_reportid_max;
	uint16_t T18_address;
	uint8_t T19_reportid;
	uint8_t T42_reportid_min;
	uint8_t T42_reportid_max;
	uint16_t T44_address;
	uint8_t T48_reportid;
	uint16_t T92_address;
	uint8_t T92_reportid;
	uint16_t T93_address;
	uint8_t T93_reportid;
	uint8_t T100_reportid_min;
	uint8_t T100_reportid_max;
	uint16_t T107_address;
	uint8_t T152_reportid_min;
	uint8_t T152_reportid_max;
	uint16_t T152_address;
	uint8_t T33_reportid_min;
	uint8_t T33_reportid_max;
	uint16_t T33_address;
	/* firmware and config update */
	char *fw_name;
	char *cfg_name;
	const char *pcfg_name;
	const char *input_name;
	char* config_file;
	struct mxt_statusinfo mxt_status;

    /* screen */
    screen_context_t screen_ctx;
    screen_window_t window;

    /* window */
    unsigned int win_height;
    unsigned int win_width;
    unsigned int win_xpos;
    unsigned int win_ypos;
    char* win_name;
    unsigned int disp_id;

	char* dev_ctrl_path;

	/* T19 keys */
	uint32_t *t19_keymap;
	unsigned int t19_num_keys;

	/* other parameters */
	unsigned int irq;
	unsigned int max_x;
	unsigned int max_y;
	bool invertx;
	bool inverty;
	bool xy_switch;
	uint8_t xsize;
	uint8_t ysize;
	bool in_bootloader;
	uint16_t mem_size;
	uint8_t t100_aux_ampl;
	uint8_t t100_aux_area;
	uint8_t t100_aux_vect;
	uint16_t T25_address;
	uint8_t  T25_reportid;
	uint8_t  t25_msg[6];
	uint8_t max_reportid;
	uint32_t config_crc;
	uint32_t info_crc;
	uint8_t bootloader_addr;
	uint8_t *msg_buf;
	uint8_t t6_status;
	bool update_input;
	uint8_t last_message_count;
	unsigned num_touchids;
	uint8_t multitouch;
	struct t7_config t7_cfg;
	unsigned long t15_keystatus;
	int t15_num_keys;
	const unsigned int *t15_keymap;
	bool use_retrigen_workaround;
	/* Indicates whether device is in suspend */
	bool suspended;
	unsigned int mtu;
	bool t25_status;
	/* startup retry options */
	unsigned max_retry_attempts;
	unsigned retry_delay;
	bool reset_expected;
	unsigned debug_enabled;
	/* connection to faceplate driver */
	int faceplate_coid;
	/* touch diagnostics */
	uint8_t touch_rtd[MAX_RTD_DATA_LENGTH];
	uint8_t touch_rtd_len;
	bool touch_rtd_readflag;
	/* knob diagnostics */
	uint8_t knob_rtd[MAX_KNOB_RTD_DATA_LENGTH];
	uint8_t knob_rtd_len;
	bool knob_rtd_readflag;

	int display_group_id;
	char* dev_status_path;
	/* verbose level */
	unsigned verbose;

	unsigned sync_point;
	/*Array for storing processed touch events */
	uint8_t touch_prev_rec[MXT_MAX_NUM_TOUCHPTS];
};

int atmel_tp_intr_attach(struct mxt_data *dev);
int mxt_create_qvm_window(struct mxt_data *dev);
void error_memory(const char * fmt, ...);


#ifdef  __cplusplus
}
#endif

#endif /* ATMEL_MXT_TS_H_ */
